<?php

namespace Simply_Static\Backup\Rest;

use Simply_Static\Backup\Backup;
use Simply_Static\Backup\Logs\Logger;
use Simply_Static\Backup\Status;

class Settings {
	/**
	 * Contains the number of failed tests.
	 *
	 * @var int
	 */
	public int $failed_tests = 0;

	/**
	 * Contains instance or null
	 *
	 * @var object|null
	 */
	private static $instance = null;

	/**
	 * Returns instance of SS_Admin_Settings.
	 *
	 * @return object
	 */
	public static function get_instance() {
		if ( null === self::$instance ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * Setting up admin fields
	 *
	 * @return void
	 */
	public function __construct() {
		add_action( 'rest_api_init', array( $this, 'rest_api_init' ) );
	}

	/**
	 * Setup Rest API endpoints.
	 *
	 * @return void
	 */
	public function rest_api_init() {
		register_rest_route( 'sss-backup-migrate/v1', '/settings', array(
			'methods'             => 'GET',
			'callback'            => [ $this, 'get_settings' ],
			'permission_callback' => function () {
				return current_user_can( apply_filters( 'ss_user_capability', 'manage_options', 'settings' ) );
			},
		) );

		register_rest_route( 'sss-backup-migrate/v1', '/settings', array(
			'methods'             => 'POST',
			'callback'            => [ $this, 'save_settings' ],
			'permission_callback' => function () {
				return current_user_can( apply_filters( 'ss_user_capability', 'manage_options', 'settings' ) );
			},
		) );

		register_rest_route( 'sss-backup-migrate/v1', '/activity-log', array(
			'methods'             => 'GET',
			'callback'            => [ $this, 'get_activity_log' ],
			'permission_callback' => function () {
				return current_user_can( apply_filters( 'ss_user_capability', 'manage_options', 'activity-log' ) );
			},
		) );

		register_rest_route( 'sss-backup-migrate/v1', '/export-log', array(
			'methods'             => 'GET',
			'callback'            => [ $this, 'get_export_log' ],
			'permission_callback' => function () {
				return current_user_can( apply_filters( 'ss_user_capability', 'manage_options', 'activity-log' ) );
			},
		) );

		register_rest_route( 'sss-backup-migrate/v1', '/start-export', array(
			'methods'             => 'POST',
			'callback'            => [ $this, 'start_export' ],
			'permission_callback' => function () {
				return current_user_can( apply_filters( 'ss_user_capability', 'publish_pages', 'generate' ) );
			},
		) );

		register_rest_route( 'sss-backup-migrate/v1', '/cancel-export', array(
			'methods'             => 'POST',
			'callback'            => [ $this, 'cancel_export' ],
			'permission_callback' => function () {
				return current_user_can( apply_filters( 'ss_user_capability', 'publish_pages', 'generate' ) );
			},
		) );

		register_rest_route( 'sss-backup-migrate/v1', '/pause-export', array(
			'methods'             => 'POST',
			'callback'            => [ $this, 'pause_export' ],
			'permission_callback' => function () {
				return current_user_can( apply_filters( 'ss_user_capability', 'publish_pages', 'generate' ) );
			},
		) );

		register_rest_route( 'sss-backup-migrate/v1', '/resume-export', array(
			'methods'             => 'POST',
			'callback'            => [ $this, 'resume_export' ],
			'permission_callback' => function () {
				return current_user_can( apply_filters( 'ss_user_capability', 'publish_pages', 'generate' ) );
			},
		) );

		register_rest_route( 'sss-backup-migrate/v1', '/is-running', array(
			'methods'             => 'GET',
			'callback'            => [ $this, 'is_running' ],
			'permission_callback' => function () {
				return current_user_can( apply_filters( 'ss_user_capability', 'publish_pages', 'generate' ) );
			},
		) );

		register_rest_route( 'sss-backup-migrate/v1', '/debug-log', array(
			'methods'             => 'GET',
			'callback'            => [ $this, 'get_debug_log' ],
			'permission_callback' => function () {
				return current_user_can( apply_filters( 'ss_user_capability', 'manage_options', 'settings' ) );
			},
		) );

		register_rest_route( 'sss-backup-migrate/v1', '/delete-backups', array(
			'methods'             => 'POST',
			'callback'            => [ $this, 'delete_all_backups' ],
			'permission_callback' => function () {
				return current_user_can( apply_filters( 'ss_user_capability', 'manage_options', 'settings' ) );
			},
		) );
	}

	/**
	 * Get settings via Rest API.
	 *
	 * @return false|mixed|null
	 */
	public function get_settings() {
		$settings = get_option( 'sss_backup_migrate' );

		return $settings;
	}

	/**
	 * Save settings via rest API.
	 *
	 * @param object $request given request.
	 *
	 * @return false|string
	 */
	public function save_settings( $request ) {
		if ( $request->get_params() ) {
			$options = sanitize_option( 'sss_backup_migrate', $request->get_params() );

			$multiline_fields = [];

			$array_fields = [];

			// Sanitize each key/value pair in options.
			foreach ( $options as $key => $value ) {
				if ( in_array( $key, $multiline_fields ) ) {
					$options[ $key ] = sanitize_textarea_field( $value );
				} elseif ( in_array( $key, $array_fields ) ) {
					$options[ $key ] = array_map( 'sanitize_text_field', $value );
				} else {
					// Exclude Basic Auth fields from sanitize.
					if ( $key === 'http_basic_auth_username' || $key === 'http_basic_auth_password' ) {
						// If they are empty, also clear $_SERVER['PHP_AUTH_USER'] and $_SERVER['PHP_AUTH_PW']
						if ( $key === 'http_basic_auth_username' && empty( $value ) ) {
							if ( isset( $_SERVER['PHP_AUTH_USER'] ) ) {
								unset( $_SERVER['PHP_AUTH_USER'] );
							}
						}

						if ( $key === 'http_basic_auth_password' && empty( $value ) ) {
							if ( isset( $_SERVER['PHP_AUTH_PW'] ) ) {
								unset( $_SERVER['PHP_AUTH_PW'] );
							}
						}

						// Continue with other options.
						continue;
					}
					$options[ $key ] = sanitize_text_field( $value );
				}
			}
			// Update settings.
			update_option( 'sss_backup_migrate', $options );

			return json_encode( [ 'status' => 200, 'message' => "Ok" ] );
		}

		return json_encode( [ 'status' => 400, 'message' => "No options updated." ] );
	}

	/**
	 * Get Activity Log.
	 *
	 * @return false|string
	 */
	public function get_activity_log( $request ) {
		$params       = $request->get_params();
		$activity_log = Status::get_messages();

		return json_encode( [
			'status'  => 200,
			'data'    => $activity_log,
			'running' => Backup::get_process()->is_active()
		] );
	}

	/**
	 * Get Backup Log
	 *
	 * @return false|string
	 */
	public function get_export_log( $request ) {
		$params     = $request->get_params();
		$export_log = Plugin::instance()->get_export_log( $params['per_page'], $params['page'], $params['blog_id'] );

		return json_encode( [
			'status' => 200,
			'data'   => $export_log,
		] );
	}

	/**
	 * Start Backup
	 *
	 * @return false|string
	 */
	public function start_export( $request ) {
		$params  = $request->get_params();
		$blog_id = ! empty( $params['blog_id'] ) ? $params['blog_id'] : 0;
		do_action( 'sssbm_before_perform_action', $blog_id, 'start' );

		// Clear previous activity log before starting new backup.
		Status::clear_messages();

		Backup::start_export();

		do_action( 'sssbm_after_perform_action', $blog_id, 'start' );

		return json_encode( [
			'status' => 200,
		] );
	}

	/**
	 * Cancel Backup
	 *
	 * @param object $request given request.
	 *
	 * @return false|string
	 */
	public function cancel_export( $request ) {
		Logger::log( "Received request to cancel static archive generation" );
		$params = $request->get_params();
		$blog_id = ! empty( $params['blog_id'] ) ? $params['blog_id'] : 0;

		do_action( 'sssbm_before_perform_action', $blog_id, 'cancel' );

		Status::error( __( 'Backup cancelled.', 'sss-backup-migrate' ) );
		Backup::clear_all_export_files();
		Backup::get_process()->cancel();
		Backup::set_as_not_running();

		do_action( 'sssbm_after_perform_action', $blog_id, 'cancel' );

		return json_encode( [ 'status' => 200 ] );
	}

	/**
	 * Is running
	 *
	 * @return false|string
	 */
	public function is_running( $request ) {
		return json_encode( [
			'status'  => 200,
			'running' => Backup::is_running() || Backup::get_process()->is_active(),
			'paused'  => Backup::get_process()->is_paused(),
			'backup'  => Backup::is_completed() && file_exists( Backup::get_backup_path() ) ? Backup::get_backup_url() : '',
		] );
	}

	/**
	 * Pause Backup
	 *
	 * @param object $request given request.
	 *
	 * @return false|string
	 */
	public function pause_export( $request ) {
		Logger::log( "Received request to pause static archive generation" );
		$params = $request->get_params();
		$blog_id = ! empty( $params['blog_id'] ) ? $params['blog_id'] : 0;

		do_action( 'sssbm_before_perform_action', $blog_id, 'pause' );

		Backup::get_process()->pause();

		do_action( 'sssbm_after_perform_action', $blog_id, 'pause' );

		return json_encode( [ 'status' => 200 ] );
	}

	/**
	 * Resume Backup
	 *
	 * @param object $request given request.
	 *
	 * @return false|string
	 */
	public function resume_export( $request ) {
		Logger::log( "Received request to resume static archive generation" );
		$params = $request->get_params();
		$blog_id = ! empty( $params['blog_id'] ) ? $params['blog_id'] : 0;

		do_action( 'sssbm_before_perform_action', $blog_id, 'resume' );

		Backup::get_process()->resume();

		do_action( 'sssbm_after_perform_action', $blog_id, 'resume' );

		return json_encode( [ 'status' => 200 ] );
	}

	/**
	 * Get Debug Log URL if it exists.
	 *
	 * @return false|string
	 */
	public function get_debug_log() {
		$debug_file = Logger::getFilename();
		$exists     = file_exists( $debug_file );
		$url        = '';

		if ( $exists ) {
			$uploads_dir = wp_upload_dir();
			$url         = $uploads_dir['baseurl'] . '/simply-static/sssbm-debug.txt';
		}

		return json_encode( [
			'status' => 200,
			'exists' => $exists,
			'url'    => $url,
		] );
	}

	/**
	 * Delete all backups from the filesystem.
	 *
	 * @return false|string
	 */
	public function delete_all_backups() {
		Backup::clear_all_export_files();

		return json_encode( [
			'status'  => 200,
			'message' => __( 'All backups deleted successfully.', 'sss-backup-migrate' ),
		] );
	}
}
