<?php

namespace Simply_Static\Backup\ThirdParty\servmask\backup;

use Simply_Static\Backup\Backup;
use Simply_Static\Backup\Helper;
use Simply_Static\Backup\Logs\Logger;
use Simply_Static\Backup\Status;

if ( ! defined( 'ABSPATH' ) ) {
	die( 'Kangaroos cannot jump here' );
}

class Backup_Content {

	public static function execute( $params = [] ) {

		// Set archive bytes offset
		if ( isset( $params['archive_bytes_offset'] ) ) {
			$archive_bytes_offset = (int) $params['archive_bytes_offset'];
		} else {
			$archive_bytes_offset = \Simply_Static\Backup\Helper::getArchiveBytes( $params );
		}

		// Set file bytes offset
		if ( isset( $params['file_bytes_offset'] ) ) {
			$file_bytes_offset = (int) $params['file_bytes_offset'];
		} else {
			$file_bytes_offset = 0;
		}

		// Set content bytes offset
		if ( isset( $params['content_bytes_offset'] ) ) {
			$content_bytes_offset = (int) $params['content_bytes_offset'];
		} else {
			$content_bytes_offset = 0;
		}

		// Get processed files size
		if ( isset( $params['content_processed_files_size'] ) ) {
			$processed_files_size = (int) $params['content_processed_files_size'];
		} else {
			$processed_files_size = 0;
		}

		// Get total content files size
		if ( isset( $params['total_content_files_size'] ) ) {
			$total_content_files_size = (int) $params['total_content_files_size'];
		} else {
			$total_content_files_size = 1;
		}

		// Get total content files count
		if ( isset( $params['total_content_files_count'] ) ) {
			$total_content_files_count = (int) $params['total_content_files_count'];
		} else {
			$total_content_files_count = 0;
		}

		// Set batch size for processing from settings (default 25 files per request)
		$settings   = get_option( 'sss_backup_migrate', [] );
		$batch_size = isset( $settings['batch_size'] ) ? absint( $settings['batch_size'] ) : 25;
		$batch_size = max( 1, $batch_size ); // Ensure minimum of 1
		$batch_size = apply_filters( 'sssbm_content_batch_size', $batch_size );

		// Get processed files count in current session
		if ( isset( $params['processed_files_count'] ) ) {
			$processed_files_count = (int) $params['processed_files_count'];
		} else {
			$processed_files_count = 0;
		}

		// Get memory limit and calculate thresholds
		$memory_limit     = self::get_memory_limit();
		$memory_threshold = $memory_limit * 0.9; // 90% threshold

		// Counter for files processed in current batch
		$current_batch_count = 0;

		// What percent of files have we processed?
		$progress = (int) min( ( $processed_files_size / $total_content_files_size ) * 100, 100 );
		Logger::log( 'Processed File size: ' . $processed_files_size . '. Total size: ' . $total_content_files_size . '. Progress: ' . $progress . '%.' );

		// Set progress
		/* translators: 1: Number of files, 2: Progress. */
		Status::info( sprintf( __( 'Backing up %1$d content files...<br />%2$d%% complete', 'sss-backup-migrate' ), $total_content_files_count, $progress ), 'ssse-content' );

		// Flag to hold if file data has been processed
		$completed = true;

		// Start time
		$start = microtime( true );

		// Get content list file
		$content_list = @fopen( Backup::get_content_list_path(), 'r' );

		// Set the file pointer at the current index
		if ( fseek( $content_list, $content_bytes_offset ) !== - 1 ) {

			// Acquire ZIP file lock to prevent concurrent access
			$zip_lock = Helper::waitForZipLock( 30 );
			if ( ! $zip_lock ) {
				Logger::log( 'Failed to acquire ZIP lock for content export' );
				@fclose( $content_list );
				$params['completed'] = false;

				return $params;
			}

			// Open the archive file for writing
			//$archive = new Compressor( ai1wm_archive_path( $params ) );
			$archive     = new \ZipArchive();
			$backup_path = Backup::get_backup_path();

			// Check if archive exists, if so open it without truncating, otherwise create it
			$zip_result = false;
			if ( file_exists( $backup_path ) ) {
				$zip_result = $archive->open( $backup_path, \ZipArchive::CHECKCONS );
				if ( $zip_result !== true ) {
					// If CHECKCONS fails, try CREATE (which won't truncate existing file)
					$zip_result = $archive->open( $backup_path, \ZipArchive::CREATE );
				}
			} else {
				$zip_result = $archive->open( $backup_path, \ZipArchive::CREATE );
			}

			// Check if ZIP file opened successfully
			if ( $zip_result !== true ) {
				Logger::log( 'Failed to open ZIP archive for content export. Error code: ' . $zip_result );
				Helper::releaseZipLock( $zip_lock );
				@fclose( $content_list );
				$params['completed'] = false;

				return $params;
			}

			// Loop over files
			while ( list( $file_abspath, $file_relpath, $file_size, $file_mtime ) = Helper::getcsv( $content_list ) ) {
				$file_bytes_written = 0;

				// Check if we've reached the batch limit
				if ( $current_batch_count >= $batch_size ) {
					Logger::log( 'Batch limit reached. Processed ' . $current_batch_count . ' files in this batch.' );
					$completed = false;
					break;
				}

				// Check memory usage before processing each file (safety check)
				$current_memory = memory_get_usage( true );
				if ( $current_memory >= $memory_threshold ) {
					Logger::log( 'Memory threshold reached. Current: ' . self::format_bytes( $current_memory ) . ', Threshold: ' . self::format_bytes( $memory_threshold ) );
					$completed = false;
					break;
				}

				// Add file to archive
				if ( ( $completed = $archive->addFile( $file_abspath, self::get_formatted_path( $file_relpath ) ) ) ) {
					$file_bytes_offset = 0;

					// Get content bytes offset
					$content_bytes_offset = ftell( $content_list );
					Logger::log( 'File: ' . $file_abspath . '. Offset: ' . $content_bytes_offset );

					$file_bytes_written = $file_size;
				} else {
					Logger::log( 'Could not add the file:' . $file_abspath );
				}

				// Increment processed files size
				$processed_files_size += $file_bytes_written;
				$processed_files_count ++;
				$current_batch_count ++;

				// What percent of files have we processed?
				$progress = (int) min( ( $processed_files_size / $total_content_files_size ) * 100, 100 );
				Logger::log( 'Processed File size: ' . $processed_files_size . '. Total size: ' . $total_content_files_size . '. Progress: ' . $progress . '%.' );
				// Set progress
				/* translators: 1: Number of files, 2: Progress. */
				Status::info( sprintf( __( 'Backing up %1$d content files...<br />%2$d%% complete', 'sss-backup-migrate' ), $total_content_files_count, $progress ), 'ssse-content' );

				// More than 10 seconds have passed, break and do another request
				if ( ( $timeout = apply_filters( 'sssbm_completed_timeout', 10 ) ) ) {
					if ( ( microtime( true ) - $start ) > $timeout ) {
						$completed = false;
						break;
					}
				}

			}

			// Close the archive file
			$archive->close();

			// Release ZIP file lock
			Helper::releaseZipLock( $zip_lock );
		}

		// Set archive bytes offset
		$params['archive_bytes_offset'] = $archive_bytes_offset;

		// Set file bytes offset
		$params['file_bytes_offset'] = $file_bytes_offset;

		// Set content bytes offset
		$params['content_bytes_offset'] = $content_bytes_offset;

		// Set processed files size
		$params['content_processed_files_size'] = $processed_files_size;

		// Set processed files count
		$params['processed_files_count'] = $processed_files_count;

		// Set total content files size
		$params['total_content_files_size'] = $total_content_files_size;

		// Set total content files count
		$params['total_content_files_count'] = $total_content_files_count;

		$params['completed'] = $completed;

		// End of the content list?
		/*if ( feof( $content_list ) ) {
			// For now, nothing here.
		}*/

		if ( $completed ) {
			Status::success( __( 'Backed up all content files...<br />100% complete', 'sss-backup-migrate' ), 'ssse-content' );;
		}

		// Close the content list file
		@fclose( $content_list );

		return $params;
	}

	/**
	 * Get the memory limit in bytes
	 *
	 * @return int Memory limit in bytes
	 */
	private static function get_memory_limit() {
		$memory_limit = ini_get( 'memory_limit' );

		if ( $memory_limit == - 1 ) {
			// No memory limit
			return PHP_INT_MAX;
		}

		// Convert memory limit to bytes
		$memory_limit = trim( $memory_limit );
		$last_char    = strtolower( $memory_limit[ strlen( $memory_limit ) - 1 ] );
		$memory_limit = (int) $memory_limit;

		switch ( $last_char ) {
			case 'g':
				$memory_limit *= 1024;
			case 'm':
				$memory_limit *= 1024;
			case 'k':
				$memory_limit *= 1024;
		}

		return $memory_limit;
	}

	public static function get_formatted_path( $path ) {

		$path_array = explode( DIRECTORY_SEPARATOR, $path );

		if ( ! in_array( 'wp-content', $path_array ) ) {
			$path_array = array_merge( [ 'wp-content' ], $path_array );
		}

		return implode( '/', $path_array );;
	}
}
